<?php

namespace App\Repositries;

use App\Http\Resources\Auction\AuctionFeatureResource;
use App\Models\Auction;
use App\Models\AuctionGallery;
use App\Models\AuctionImage;
use App\Models\FeatureAttribute;

class  AuctionRepository
{
    public $mainRepositry;
    public $model;

    public function __construct(MainRepository $mainRepositry, Auction $model)
    {
        $this->mainRepositry = $mainRepositry;
        $this->model         = $model;
    }

    public function index($filters = [])
    {
        if (auth()->guard('client')->check() && !empty($filters['search'])) {
            $this->mainRepositry->create(auth('client')->user()->searches(), [
                'search' => $filters['search']
            ]);
        }
        $autions = $this->model->query()->filter($filters)->with(['country', 'city', 'area', 'agency']);
        return $this->mainRepositry->index($autions);
    }

    public function show($id)
    {
        $auction = $this->mainRepositry->show($this->model, $id);
        $auction = $auction->load(['images', 'gallery', 'category', 'country', 'city', 'area', 'agency']);

        if (!auth()->guard('web')->check()) {
            $auction['feature_attributes'] = AuctionFeatureResource::collection($auction->features->unique('feature_id'));
            $auction['related'] = $this->related($auction);
        } else {
            $auction->load('features');
        }
        return $auction;
    }

    public function store($data)
    {
        $data = $this->isHome($data);
        $auction  = $this->mainRepositry->create($this->model, $data);
        $this->createImages($auction, $data['images'] ?? []);
        $this->createGallery($auction, $data['gallery'] ?? []);
        $this->createFeatures($auction, $data['features']);
        return $auction;
    }

    public function update($id, $data)
    {
        $data = $this->isHome($data);
        $auction  = $this->mainRepositry->update($this->model, $id, $data);
        $this->createImages($auction, $data['images'] ?? []);
        $this->createGallery($auction, $data['gallery'] ?? []);
        $this->createFeatures($auction, $data['features'] ?? []);
        return $auction;
    }

    public function destroy($id)
    {
        $auction = $this->mainRepositry->show($this->model, $id);
        foreach ($auction->images as $image) {
            $this->deleteImage($image->id);
        }
        foreach ($auction->gallery as $gallery) {
            $this->deleteGallery($gallery->id);
        }
        $this->mainRepositry->destroy($this->model, $id);
    }

    public function createImages($auction, $images = [])
    {
        if (!empty($images)) {
            foreach ($images as  $image) {
                $auction->images()->create([
                    'image' => $image
                ]);
            }
        }
    }

    public function createGallery($auction, $gallery = [])
    {
        if (!empty($gallery)) {
            foreach ($gallery as  $item) {
                $auction->gallery()->create([
                    'image' => $item
                ]);
            }
        }
    }

    public function createFeatures($auction, $features = [])
    {
        if (!empty($features)) {
            foreach ($features as  $feature) {
                $this->mainRepositry->show(FeatureAttribute::where('feature_id', $feature['feature_id']), $feature['feature_attribute_id']);
                $auction->features()->updateOrCreate(
                    [
                        'id' => $feature['id'] ?? Null
                    ],
                    $feature
                );
            }
        }
    }

    public function deleteImage($id)
    {
        $this->mainRepositry->destroy(new AuctionImage(), $id);
    }

    public function deleteGallery($id)
    {
        $this->mainRepositry->destroy(new AuctionGallery(), $id);
    }

    private function isHome($data)
    {
        $data['is_home'] =  isset($data['is_home']) && $data['is_home'] ? true : false;
        return $data;
    }

    public function dates()
    {
        return $this->mainRepositry->index($this->model->select('date')->distinct())->pluck('date')->toArray();
    }

    public function map($filters = [])
    {
        return $this->mainRepositry->index($this->model->query()->filter($filters)->select('lat', 'long'))->toArray();
    }

    public function related($auction)
    {
        $filters = [
            'except_id' => $auction->id,
            'category_id' => $auction->category_id,
            'city_id' => $auction->city_id,
            'agency_id' => $auction->agency_id
        ];
        return  $this->index($filters)->load('images', 'category', 'agency', 'country', 'city', 'area')->sortByDesc('date')->take(10)->values();
    }

    public function getMap($auction)
    {
        return $this->related($auction)->select('lat', 'long');
    }

    public function expired()
    {
        return $this->mainRepositry->index($this->model->expired());
    }

    public function start()
    {
        return $this->mainRepositry->index($this->model->start());
    }
}
