<?php

namespace App\Models;

use App\Http\Traits\ImageTrait;
use App\Http\Traits\PasswordTrait;
use App\Http\Traits\StatusTrait;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Tymon\JWTAuth\Contracts\JWTSubject;


class Client extends Authenticatable implements JWTSubject
{
    use PasswordTrait, StatusTrait, ImageTrait;

    public $folder = 'clients';
    protected $fillable = [
        'name',
        'email',
        'phone',
        'password',
        'status',
        'fcm_token',
        'image',
        'provider',
        'verified_at',
        'verification_code',
        'verify_expire'
    ];

    protected $hidden = ['password'];

    public static function rules()
    {
        return [
            "name"              => 'required|max:255',
            "email"             => 'required|email|max:255|unique:clients,email,',
            'phone'             => 'required|numeric|regex:/^([0-9\s\-\+\(\)]*)$/|unique:clients,phone,',
            'password'          => 'same:confirm_password|min:8|',
            'confirm_password' => 'required_with:password',
            'image'             => 'sometimes|image|mimes:jpg,jpeg,png|max:3072',
        ];
    }

    protected $appends = ['status_text'];

    protected function casts(): array
    {
        return [
            'password' => 'hashed',
        ];
    }

    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    public function getJWTCustomClaims()
    {
        return [];
    }

    public function otps()
    {
        return $this->hasMany(Otp::class);
    }


    public function favourites()
    {
        return $this->belongsToMany(Auction::class, 'favourites', 'client_id', 'auction_id')->withPivot('id')->withTimestamps();
    }

    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }

    public function searches()
    {
        return $this->hasMany(RecentSearch::class);
    }
}
